import { Link, useLocation } from "wouter";
import { cn } from "@/lib/utils";
import { motion } from "framer-motion";
import { Menu, X } from "lucide-react";
import { useState } from "react";

const links = [
  { href: "/", label: "Home" },
  { href: "/work", label: "Work" },
  { href: "/services", label: "Services" },
  { href: "/about", label: "About" },
  { href: "/a1inc-2", label: "A1INC 2.0", special: true },
];

export function Navbar() {
  const [location] = useLocation();
  const [mobileMenuOpen, setMobileMenuOpen] = useState(false);

  return (
    <>
      <motion.nav 
        initial={{ y: -100, opacity: 0 }}
        animate={{ y: 0, opacity: 1 }}
        transition={{ duration: 0.5, ease: "circOut" }}
        className="fixed top-0 left-0 right-0 z-50 flex justify-center pt-6 px-4"
      >
        <div className="glass-1 rounded-full px-6 py-3 flex items-center justify-between gap-8 max-w-5xl w-full">
          {/* Logo */}
          <Link href="/" className="font-display font-bold text-2xl tracking-tighter hover:opacity-80 transition-opacity">
            A1INC
          </Link>

          {/* Desktop Links */}
          <div className="hidden md:flex items-center gap-1 bg-black/20 rounded-full px-2 py-1 border border-white/5">
            {links.map((link) => (
              <Link 
                key={link.href} 
                href={link.href}
                className={cn(
                  "px-4 py-2 rounded-full text-sm font-medium transition-all duration-300 relative",
                  location === link.href ? "text-black" : "text-white/60 hover:text-white",
                  link.special && location !== link.href && "text-purple-400 hover:text-purple-300"
                )}
              >
                {location === link.href && (
                  <motion.div
                    layoutId="navbar-indicator"
                    className="absolute inset-0 bg-white rounded-full"
                    initial={false}
                    transition={{ type: "spring", stiffness: 300, damping: 30 }}
                  />
                )}
                <span className="relative z-10">{link.label}</span>
              </Link>
            ))}
          </div>

          {/* Right CTA */}
          <div className="hidden md:flex items-center">
            <Link 
              href="/contact"
              className="group relative overflow-hidden rounded-full bg-white px-6 py-2 text-sm font-semibold text-black transition-transform hover:scale-105 active:scale-95"
            >
              Start Project
            </Link>
          </div>

          {/* Mobile Toggle */}
          <button 
            className="md:hidden text-white"
            onClick={() => setMobileMenuOpen(!mobileMenuOpen)}
          >
            {mobileMenuOpen ? <X /> : <Menu />}
          </button>
        </div>
      </motion.nav>

      {/* Mobile Menu */}
      {mobileMenuOpen && (
        <motion.div 
          initial={{ opacity: 0, y: -20 }}
          animate={{ opacity: 1, y: 0 }}
          exit={{ opacity: 0, y: -20 }}
          className="fixed inset-0 z-40 bg-black/95 backdrop-blur-xl pt-32 px-6 md:hidden"
        >
          <div className="flex flex-col gap-6 items-center">
            {links.map((link) => (
              <Link 
                key={link.href} 
                href={link.href}
                className={cn(
                  "text-2xl font-display font-medium",
                  location === link.href ? "text-white" : "text-white/50"
                )}
                onClick={() => setMobileMenuOpen(false)}
              >
                {link.label}
              </Link>
            ))}
            <Link 
              href="/contact"
              className="mt-8 bg-white text-black px-8 py-3 rounded-full font-bold text-lg"
              onClick={() => setMobileMenuOpen(false)}
            >
              Start Project
            </Link>
          </div>
        </motion.div>
      )}
    </>
  );
}
