import { useQuery, useMutation, useQueryClient } from "@tanstack/react-query";
import { api, buildUrl } from "@shared/routes";
import { type InsertContactSubmission } from "@shared/schema";
import { useToast } from "@/hooks/use-toast";

// ============================================
// WORK (Case Studies)
// ============================================

export function useCaseStudies() {
  return useQuery({
    queryKey: [api.work.list.path],
    queryFn: async () => {
      const res = await fetch(api.work.list.path);
      if (!res.ok) throw new Error("Failed to fetch case studies");
      return api.work.list.responses[200].parse(await res.json());
    },
  });
}

export function useCaseStudy(slug: string) {
  return useQuery({
    queryKey: [api.work.get.path, slug],
    queryFn: async () => {
      const url = buildUrl(api.work.get.path, { slug });
      const res = await fetch(url);
      if (!res.ok) {
        if (res.status === 404) return null;
        throw new Error("Failed to fetch case study");
      }
      return api.work.get.responses[200].parse(await res.json());
    },
    enabled: !!slug,
  });
}

// ============================================
// SERVICES
// ============================================

export function useServices() {
  return useQuery({
    queryKey: [api.services.list.path],
    queryFn: async () => {
      const res = await fetch(api.services.list.path);
      if (!res.ok) throw new Error("Failed to fetch services");
      return api.services.list.responses[200].parse(await res.json());
    },
  });
}

// ============================================
// CONTACT
// ============================================

export function useSubmitContact() {
  const { toast } = useToast();
  
  return useMutation({
    mutationFn: async (data: InsertContactSubmission) => {
      // Validate input manually if needed, or rely on API response validation
      const validated = api.contact.submit.input.parse(data);
      
      const res = await fetch(api.contact.submit.path, {
        method: api.contact.submit.method,
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify(validated),
      });

      if (!res.ok) {
        if (res.status === 400) {
          const error = api.contact.submit.responses[400].parse(await res.json());
          throw new Error(error.message);
        }
        throw new Error("Failed to submit inquiry");
      }
      
      return api.contact.submit.responses[201].parse(await res.json());
    },
    onSuccess: () => {
      toast({
        title: "Message Sent",
        description: "We'll be in touch shortly.",
      });
    },
    onError: (error) => {
      toast({
        title: "Error",
        description: error.message,
        variant: "destructive",
      });
    },
  });
}
