import { PageTransition } from "@/components/PageTransition";
import { useCaseStudy } from "@/hooks/use-a1inc";
import { useRoute } from "wouter";
import { Loader2, ArrowLeft, CheckCircle2 } from "lucide-react";
import { Link } from "wouter";
import { GlassCard } from "@/components/GlassCard";

export default function CaseStudy() {
  const [, params] = useRoute("/work/:slug");
  const slug = params?.slug || "";
  const { data: project, isLoading } = useCaseStudy(slug);

  if (isLoading) return <div className="h-screen flex items-center justify-center"><Loader2 className="animate-spin text-white/20" /></div>;
  if (!project) return <div className="h-screen flex items-center justify-center text-white/40">Project not found</div>;

  const content = project.content as any;

  return (
    <PageTransition>
      {/* Hero Image */}
      <div className="w-full h-[60vh] relative overflow-hidden">
        <div className="absolute inset-0 bg-gradient-to-t from-background to-transparent z-10" />
        <img src={project.heroImage} alt={project.title} className="w-full h-full object-cover" />
      </div>

      <div className="max-w-4xl mx-auto px-6 -mt-32 relative z-20">
        <Link href="/work" className="inline-flex items-center gap-2 text-white/50 hover:text-white mb-8 transition-colors">
          <ArrowLeft className="w-4 h-4" /> Back to Work
        </Link>
        
        <h1 className="text-5xl md:text-7xl font-display font-bold mb-6">{project.title}</h1>
        <p className="text-xl md:text-2xl text-white/70 leading-relaxed mb-12">{project.summary}</p>

        {/* Meta Grid */}
        <div className="grid grid-cols-2 md:grid-cols-4 gap-6 border-y border-white/10 py-8 mb-16">
          <div>
            <h4 className="text-xs uppercase tracking-wider text-white/40 mb-2">Role</h4>
            <p className="font-medium">{project.role}</p>
          </div>
          <div>
            <h4 className="text-xs uppercase tracking-wider text-white/40 mb-2">Timeline</h4>
            <p className="font-medium">{project.timeline}</p>
          </div>
          <div>
            <h4 className="text-xs uppercase tracking-wider text-white/40 mb-2">Category</h4>
            <p className="font-medium">{project.category}</p>
          </div>
          <div>
            <h4 className="text-xs uppercase tracking-wider text-white/40 mb-2">Stack</h4>
            <div className="flex flex-wrap gap-2">
              {project.stack.map(tech => (
                <span key={tech} className="bg-white/10 px-2 py-0.5 rounded text-xs">{tech}</span>
              ))}
            </div>
          </div>
        </div>

        {/* Highlights Section */}
        {content.highlights && (
          <div className="grid grid-cols-1 md:grid-cols-3 gap-4 mb-20">
            {content.highlights.map((h: string, idx: number) => (
              <GlassCard key={idx} className="p-6 text-center">
                <p className="text-3xl font-bold mb-1">{h.split(' ')[0]}</p>
                <p className="text-xs uppercase tracking-widest text-white/40">{h.split(' ').slice(1).join(' ')}</p>
              </GlassCard>
            ))}
          </div>
        )}

        {/* Content Sections */}
        <div className="space-y-20">
          {content.sections && content.sections.map((section: any, idx: number) => (
            <div key={idx} className="grid grid-cols-1 md:grid-cols-3 gap-8">
              <h3 className="text-2xl font-bold font-display text-white/80">{section.title}</h3>
              <div className="md:col-span-2 prose prose-invert prose-lg text-white/60">
                <p>{section.body}</p>
                {section.image && (
                  <GlassCard className="mt-8 p-2 bg-white/5 border-0">
                    <img src={section.image} alt={section.title} className="rounded-lg w-full" />
                  </GlassCard>
                )}
              </div>
            </div>
          ))}
        </div>

        {/* Result Detail Block */}
        <div className="mt-32">
          <GlassCard className="p-12 border-white/5 bg-white/[0.02]">
             <h3 className="text-3xl font-bold mb-8">Key Deliverables</h3>
             <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                {[
                  "High-fidelity Design System",
                  "Performance-optimized Frontend",
                  "Robust Backend Architecture",
                  "Comprehensive Security Audit"
                ].map((item, i) => (
                  <div key={i} className="flex items-center gap-4 text-white/60">
                    <CheckCircle2 className="w-5 h-5 text-white/20" />
                    <span>{item}</span>
                  </div>
                ))}
             </div>
          </GlassCard>
        </div>

        {/* Next Project CTA (Simple) */}
        <div className="mt-32 py-16 border-t border-white/5 text-center">
          <p className="text-white/40 mb-4">Like what you see?</p>
          <Link href="/contact" className="text-4xl font-display font-bold hover:text-white/60 transition-colors">
            Start a Project with Us →
          </Link>
        </div>
      </div>
    </PageTransition>
  );
}
