import { PageTransition } from "@/components/PageTransition";
import { GlassCard } from "@/components/GlassCard";
import { useSubmitContact } from "@/hooks/use-a1inc";
import { useForm } from "react-hook-form";
import { zodResolver } from "@hookform/resolvers/zod";
import { insertContactSchema, type InsertContactSubmission } from "@shared/schema";
import { Form, FormControl, FormField, FormItem, FormLabel, FormMessage } from "@/components/ui/form";
import { Input } from "@/components/ui/input";
import { Textarea } from "@/components/ui/textarea";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Button } from "@/components/ui/button";
import { Loader2, Mail, MapPin, Phone } from "lucide-react";

export default function Contact() {
  const mutation = useSubmitContact();
  
  const form = useForm<InsertContactSubmission>({
    resolver: zodResolver(insertContactSchema),
    defaultValues: {
      name: "",
      email: "",
      company: "",
      budget: "",
      message: ""
    }
  });

  function onSubmit(data: InsertContactSubmission) {
    mutation.mutate(data, {
      onSuccess: () => form.reset()
    });
  }

  return (
    <PageTransition>
      <div className="max-w-7xl mx-auto px-6 py-20 min-h-screen flex flex-col justify-center">
        <div className="grid grid-cols-1 lg:grid-cols-2 gap-16 mb-32">
          <div className="space-y-8">
            <h1 className="text-5xl md:text-7xl font-display font-bold">
              Let's Build <br/> The Future.
            </h1>
            <p className="text-xl text-white/60 max-w-md">
              Ready to elevate your digital presence? Tell us about your project and we'll engineer the perfect solution.
            </p>
            
            <div className="space-y-6 pt-8">
              <div className="flex items-center gap-6 text-white/40 group">
                <div className="w-12 h-12 rounded-full border border-white/10 flex items-center justify-center group-hover:border-white/30 transition-colors">
                  <MapPin className="w-5 h-5" />
                </div>
                <div>
                  <p className="text-xs uppercase tracking-widest mb-1">Location</p>
                  <span className="text-white/80">San Francisco, CA</span>
                </div>
              </div>
              <div className="flex items-center gap-6 text-white/40 group">
                <div className="w-12 h-12 rounded-full border border-white/10 flex items-center justify-center group-hover:border-white/30 transition-colors">
                  <Mail className="w-5 h-5" />
                </div>
                <div>
                  <p className="text-xs uppercase tracking-widest mb-1">Email</p>
                  <span className="text-white/80">hello@a1inc.site</span>
                </div>
              </div>
            </div>
          </div>

          <GlassCard className="p-8 md:p-12 border-white/10 bg-white/5">
            <Form {...form}>
              <form onSubmit={form.handleSubmit(onSubmit)} className="space-y-6">
                <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                  <FormField
                    control={form.control}
                    name="name"
                    render={({ field }) => (
                      <FormItem>
                        <FormLabel>Name</FormLabel>
                        <FormControl>
                          <Input placeholder="John Doe" className="bg-black/20 border-white/10 focus:border-white/30 h-12" {...field} />
                        </FormControl>
                        <FormMessage />
                      </FormItem>
                    )}
                  />
                  <FormField
                    control={form.control}
                    name="email"
                    render={({ field }) => (
                      <FormItem>
                        <FormLabel>Email</FormLabel>
                        <FormControl>
                          <Input placeholder="john@company.com" className="bg-black/20 border-white/10 focus:border-white/30 h-12" {...field} />
                        </FormControl>
                        <FormMessage />
                      </FormItem>
                    )}
                  />
                </div>

                <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                  <FormField
                    control={form.control}
                    name="company"
                    render={({ field }) => (
                      <FormItem>
                        <FormLabel>Company</FormLabel>
                        <FormControl>
                          <Input placeholder="Acme Inc." className="bg-black/20 border-white/10 focus:border-white/30 h-12" {...field} value={field.value || ''} />
                        </FormControl>
                        <FormMessage />
                      </FormItem>
                    )}
                  />
                  <FormField
                    control={form.control}
                    name="budget"
                    render={({ field }) => (
                      <FormItem>
                        <FormLabel>Budget Range</FormLabel>
                        <Select onValueChange={field.onChange} defaultValue={field.value || undefined}>
                          <FormControl>
                            <SelectTrigger className="bg-black/20 border-white/10 focus:border-white/30 h-12">
                              <SelectValue placeholder="Select a range" />
                            </SelectTrigger>
                          </FormControl>
                          <SelectContent className="bg-zinc-900 border-white/10 text-white">
                            <SelectItem value="$10k - $25k">$10k - $25k</SelectItem>
                            <SelectItem value="$25k - $50k">$25k - $50k</SelectItem>
                            <SelectItem value="$50k - $100k">$50k - $100k</SelectItem>
                            <SelectItem value="$100k+">$100k+</SelectItem>
                          </SelectContent>
                        </Select>
                        <FormMessage />
                      </FormItem>
                    )}
                  />
                </div>

                <FormField
                  control={form.control}
                  name="message"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>Project Details</FormLabel>
                      <FormControl>
                        <Textarea 
                          placeholder="Tell us about your goals..." 
                          className="bg-black/20 border-white/10 focus:border-white/30 min-h-[150px] resize-none" 
                          {...field} 
                        />
                      </FormControl>
                      <FormMessage />
                    </FormItem>
                  )}
                />

                <Button 
                  type="submit" 
                  disabled={mutation.isPending}
                  className="w-full h-12 rounded-lg bg-white text-black font-bold hover:bg-white/90 transition-colors"
                >
                  {mutation.isPending ? (
                    <><Loader2 className="mr-2 h-4 w-4 animate-spin" /> Sending...</>
                  ) : (
                    "Submit Inquiry"
                  )}
                </Button>
              </form>
            </Form>
          </GlassCard>
        </div>

        {/* Expansion Blocks */}
        <div className="grid grid-cols-1 md:grid-cols-3 gap-8">
           <GlassCard className="p-8">
             <h3 className="text-xl font-bold mb-4">Direct Communication</h3>
             <p className="text-white/50 text-sm leading-relaxed">
               Skip the account managers. Work directly with the engineers and designers building your project.
             </p>
           </GlassCard>
           <GlassCard className="p-8 bg-white/5">
             <h3 className="text-xl font-bold mb-4">Global Network</h3>
             <p className="text-white/50 text-sm leading-relaxed">
               With a distributed team, we operate across timezones to ensure constant momentum on your build.
             </p>
           </GlassCard>
           <GlassCard className="p-8">
             <h3 className="text-xl font-bold mb-4">Long-term Partnership</h3>
             <p className="text-white/50 text-sm leading-relaxed">
               We don't just launch and leave. We offer ongoing maintenance and strategic scaling support.
             </p>
           </GlassCard>
        </div>
      </div>
    </PageTransition>
  );
}
