import { PageTransition } from "@/components/PageTransition";
import { GlassCard } from "@/components/GlassCard";
import { Signature } from "@/components/Signature";
import { motion } from "framer-motion";
import { ArrowRight, Zap, Layers, Cpu, Globe, Target, Shield, Smartphone } from "lucide-react";
import { Link } from "wouter";
import heroVideo from "@/assets/videos/hero-bg.mp4";

export default function Home() {
  return (
    <PageTransition>
      {/* HERO SECTION */}
      <section className="relative min-h-[90vh] flex flex-col justify-center items-center px-6 overflow-hidden">
        {/* Video Background */}
        <div className="absolute inset-0 z-0">
          <video
            autoPlay
            loop
            muted
            playsInline
            className="w-full h-full object-cover opacity-40"
          >
            <source src={heroVideo} type="video/mp4" />
          </video>
          {/* Dark Wash Gradient */}
          <div className="absolute inset-0 bg-gradient-to-b from-black/60 via-black/20 to-black/60 z-10" />
        </div>

        {/* Abstract Background Element (Keeping for additional depth) */}
        <div className="absolute top-1/2 left-1/2 -translate-x-1/2 -translate-y-1/2 w-[800px] h-[800px] bg-white/[0.02] rounded-full blur-3xl pointer-events-none z-10" />
        
        <motion.div 
          initial={{ opacity: 0, y: 30 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.8, delay: 0.2 }}
          className="relative z-20 text-center space-y-8 max-w-4xl"
        >
          <div className="inline-flex items-center gap-2 px-3 py-1 rounded-full border border-white/10 bg-white/5 text-xs tracking-widest uppercase font-medium text-white/60 mb-4">
            <span className="w-1.5 h-1.5 rounded-full bg-green-500 animate-pulse" />
            Accepting New Projects
          </div>
          
          <h1 className="text-6xl md:text-8xl lg:text-9xl font-bold leading-[0.9] tracking-tighter text-transparent bg-clip-text bg-gradient-to-b from-white to-white/40">
            Powering Digital <br /> Systems
          </h1>
          
          <p className="text-lg md:text-xl text-white/60 max-w-2xl mx-auto leading-relaxed">
            Design-led experiences, engineered for scale and longevity.
          </p>
          
          <div className="flex flex-col sm:flex-row gap-4 justify-center pt-8">
            <Link href="/work" className="glass-1 px-8 py-4 rounded-full font-medium hover:bg-white/10 transition-colors flex items-center gap-2 group">
              View Our Work
              <ArrowRight className="w-4 h-4 transition-transform group-hover:translate-x-1" />
            </Link>
            <Link href="/contact" className="px-8 py-4 rounded-full font-medium bg-primary text-primary-foreground hover:bg-primary/90 transition-colors">
              Start a Project
            </Link>
          </div>
        </motion.div>
      </section>

      {/* DIFFERENCE SECTION */}
      <section className="py-32 px-6 max-w-7xl mx-auto">
        <div className="grid grid-cols-1 md:grid-cols-2 gap-8 items-center">
          <div className="space-y-6">
            <h2 className="text-4xl md:text-5xl font-bold">Why A1INC?</h2>
            <p className="text-white/60 text-lg leading-relaxed">
              Most agencies stop at design. We start there. Our engineering-first approach ensures your digital presence is not just beautiful, but scalable, performant, and built for growth.
            </p>
          </div>
          <div className="grid gap-4">
            <GlassCard className="flex items-center justify-between group cursor-default hover:bg-white/5">
              <span className="text-white/40 group-hover:text-white transition-colors">Traditional Agencies</span>
              <span className="text-red-400">Surface Level</span>
            </GlassCard>
            <GlassCard className="flex items-center justify-between border-white/20 bg-white/10">
              <span className="font-semibold">A1INC Approach</span>
              <span className="text-green-400">Deep Engineering</span>
            </GlassCard>
          </div>
        </div>
      </section>

      {/* NEW: MISSION SECTION */}
      <section className="py-32 px-6 max-w-7xl mx-auto">
        <div className="grid grid-cols-1 md:grid-cols-3 gap-8">
          <div className="md:col-span-2">
            <h2 className="text-4xl font-bold mb-8">Our Mission</h2>
            <p className="text-2xl text-white/40 leading-relaxed font-display">
              To bridge the gap between <span className="text-white">imagination</span> and <span className="text-white">execution</span> through high-precision digital craftsmanship.
            </p>
          </div>
          <GlassCard className="p-8 flex flex-col justify-between aspect-square md:aspect-auto">
            <Target className="w-12 h-12 text-white/20" />
            <div>
              <h3 className="text-xl font-bold mb-2">Zero Compromise</h3>
              <p className="text-sm text-white/50">Every line of code and every pixel is audited for maximum impact.</p>
            </div>
          </GlassCard>
        </div>
      </section>

      {/* WORKFLOW SECTION */}
      <section className="py-32 px-6 border-y border-white/5 bg-white/[0.02]">
        <div className="max-w-7xl mx-auto">
          <h2 className="text-3xl font-bold mb-16 text-center">Our Process</h2>
          <div className="grid grid-cols-1 md:grid-cols-3 gap-8">
            {[
              { num: "01", title: "Strategy", desc: "We define the core problem and chart the most efficient path to solution." },
              { num: "02", title: "Design", desc: "Visualizing the future state with high-fidelity prototypes and systems." },
              { num: "03", title: "Engineering", desc: "Production-grade code, rigorous testing, and seamless deployment." }
            ].map((step, i) => (
              <motion.div 
                key={i}
                initial={{ opacity: 0, y: 20 }}
                whileInView={{ opacity: 1, y: 0 }}
                viewport={{ once: true }}
                transition={{ delay: i * 0.2 }}
                className="relative p-8 group"
              >
                <span className="text-8xl font-bold text-white/[0.03] absolute -top-4 -left-4 select-none group-hover:text-white/[0.06] transition-colors">{step.num}</span>
                <h3 className="text-xl font-bold mb-4 relative z-10">{step.title}</h3>
                <p className="text-white/50 relative z-10">{step.desc}</p>
              </motion.div>
            ))}
          </div>
        </div>
      </section>

      {/* CAPABILITIES GRID */}
      <section className="py-32 px-6 max-w-7xl mx-auto">
        <h2 className="text-4xl font-bold mb-12">Capabilities</h2>
        <div className="grid grid-cols-1 md:grid-cols-3 gap-4 h-auto md:h-[600px]">
          <GlassCard className="md:col-span-2 md:row-span-2 relative overflow-hidden group">
            <div className="absolute inset-0 bg-gradient-to-br from-purple-900/20 to-transparent opacity-0 group-hover:opacity-100 transition-opacity duration-500" />
            <div className="relative z-10 h-full flex flex-col justify-end">
              <Layers className="w-12 h-12 mb-6 text-white/80" />
              <h3 className="text-2xl font-bold mb-2">Design Systems</h3>
              <p className="text-white/60">Scalable UI libraries that ensure consistency across all touchpoints.</p>
            </div>
          </GlassCard>
          <GlassCard className="group hover:bg-white/10 transition-colors">
            <Zap className="w-8 h-8 mb-4 text-yellow-400" />
            <h3 className="text-xl font-bold mb-2">High Performance</h3>
            <p className="text-sm text-white/50">Next.js & React optimization.</p>
          </GlassCard>
          <GlassCard className="group hover:bg-white/10 transition-colors">
            <Cpu className="w-8 h-8 mb-4 text-blue-400" />
            <h3 className="text-xl font-bold mb-2">AI Integration</h3>
            <p className="text-sm text-white/50">LLM & Automation workflows.</p>
          </GlassCard>
          <GlassCard className="md:col-span-2 flex items-center gap-6 group hover:bg-white/10 transition-colors">
            <Globe className="w-12 h-12 text-green-400" />
            <div>
              <h3 className="text-xl font-bold mb-1">Global Scale</h3>
              <p className="text-sm text-white/50">Edge networking and distributed architecture.</p>
            </div>
          </GlassCard>
        </div>
      </section>

      {/* NEW: SECURITY & MOBILITY SECTION */}
      <section className="py-32 px-6 max-w-7xl mx-auto border-t border-white/5">
        <div className="grid grid-cols-1 md:grid-cols-2 gap-16 items-center">
           <GlassCard className="p-12 relative overflow-hidden">
             <div className="relative z-10">
               <Shield className="w-16 h-16 mb-8 text-white/20" />
               <h3 className="text-3xl font-bold mb-4">Bulletproof Security</h3>
               <p className="text-white/50">
                 We implement enterprise-grade security protocols from day one. Data integrity and user privacy are engineered into the core of every solution.
               </p>
             </div>
           </GlassCard>
           <div className="space-y-8">
             <div className="flex gap-6 items-start">
               <Smartphone className="w-8 h-8 text-white/40 shrink-0" />
               <div>
                 <h4 className="text-xl font-bold mb-2">Mobile First</h4>
                 <p className="text-white/50">Experiences that feel native on every device, optimized for high-speed delivery.</p>
               </div>
             </div>
             <div className="flex gap-6 items-start">
               <Layers className="w-8 h-8 text-white/40 shrink-0" />
               <div>
                 <h4 className="text-xl font-bold mb-2">Technical Debt Elimination</h4>
                 <p className="text-white/50">Refactoring legacy systems into modern, efficient architectures.</p>
               </div>
             </div>
           </div>
        </div>
      </section>

    </PageTransition>
  );
}
