import { useState } from "react";
import { PageTransition } from "@/components/PageTransition";
import { useServices } from "@/hooks/use-a1inc";
import { GlassCard } from "@/components/GlassCard";
import { 
  Loader2, Code, Paintbrush, Monitor, Rocket, Cpu, 
  BarChart, Settings, Search, Video, CheckCircle2, 
  ArrowRight
} from "lucide-react";
import { motion } from "framer-motion";
import { Button } from "@/components/ui/button";
import { Dialog, DialogContent, DialogHeader, DialogTitle } from "@/components/ui/dialog";

const iconMap: Record<string, any> = {
  "Code": Code,
  "Paintbrush": Paintbrush,
  "Monitor": Monitor,
  "Rocket": Rocket,
  "Cpu": Cpu,
  "BarChart": BarChart,
  "Settings": Settings,
  "Search": Search,
  "Video": Video
};

export default function Services() {
  const { data: services, isLoading } = useServices();
  const [selectedService, setSelectedService] = useState<any>(null);

  const processSteps = [
    { title: "Consultation", desc: "Discover goals and challenges." },
    { title: "Planning", desc: "Tailored strategy and roadmap." },
    { title: "Implementation", desc: "Execution with precision." },
    { title: "Analysis", desc: "Insight gathering and performance tracking." },
    { title: "Optimization", desc: "Ongoing refinement for growth." },
    { title: "Growth Partnership", desc: "Long-term digital evolution support." }
  ];

  return (
    <PageTransition>
      <div className="max-w-7xl mx-auto px-6 py-20">
        {/* Hero Section */}
        <div className="text-center mb-32 max-w-3xl mx-auto">
          <motion.h1 
            initial={{ opacity: 0, y: 20 }}
            animate={{ opacity: 1, y: 0 }}
            className="text-5xl md:text-7xl font-display font-bold mb-8 leading-tight"
          >
            Digital Services <br/>
            <span className="text-white/40 italic font-light">Built for Growth</span>
          </motion.h1>
          <motion.p 
            initial={{ opacity: 0, y: 20 }}
            animate={{ opacity: 1, y: 0 }}
            transition={{ delay: 0.1 }}
            className="text-lg text-white/60 mb-10"
          >
            Elevating digital presence with high-performance web systems, creative strategy, and data-driven execution.
          </motion.p>
          <motion.div
            initial={{ opacity: 0, y: 20 }}
            animate={{ opacity: 1, y: 0 }}
            transition={{ delay: 0.2 }}
          >
            <Button size="lg" className="rounded-full px-8 h-14 text-base font-medium">
              Get a Free Consultation
              <ArrowRight className="ml-2 w-4 h-4" />
            </Button>
          </motion.div>
        </div>

        {/* Core Services Overview */}
        <div className="mb-40">
          <div className="flex flex-col md:flex-row justify-between items-end mb-16 gap-8">
            <div className="max-w-xl">
              <h2 className="text-sm font-mono text-white/40 uppercase tracking-[0.2em] mb-4">Core Expertise</h2>
              <h3 className="text-3xl md:text-4xl font-bold">What We Do Best</h3>
              <p className="text-white/50 mt-4">A1INC delivers a full suite of digital services tailored to modern brands that value both aesthetics and performance.</p>
            </div>
          </div>

          {isLoading ? (
            <div className="flex justify-center py-20">
              <Loader2 className="animate-spin w-8 h-8 text-white/20" />
            </div>
          ) : (
            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
              {services?.map((service, idx) => {
                const Icon = iconMap[service.icon] || Code;
                return (
                  <motion.div
                    key={service.id}
                    initial={{ opacity: 0, y: 20 }}
                    animate={{ opacity: 1, y: 0 }}
                    transition={{ delay: idx * 0.1 }}
                    onClick={() => setSelectedService(service)}
                  >
                    <GlassCard className="h-full group hover:bg-white/10 transition-all cursor-pointer border-white/5 p-8">
                      <div className="w-14 h-14 rounded-2xl bg-white/5 flex items-center justify-center mb-8 group-hover:scale-110 group-hover:bg-white/10 transition-all">
                        <Icon className="w-7 h-7 text-white/80" />
                      </div>
                      <div className="text-xs font-mono text-white/40 mb-3 uppercase tracking-wider">{service.category}</div>
                      <h3 className="text-2xl font-bold mb-4">{service.title}</h3>
                      <p className="text-white/60 text-sm leading-relaxed mb-8">{service.description}</p>
                      <div className="mt-auto flex items-center text-sm font-medium text-white/40 group-hover:text-white transition-colors">
                        Learn More
                        <ArrowRight className="ml-2 w-4 h-4 opacity-0 -translate-x-2 group-hover:opacity-100 group-hover:translate-x-0 transition-all" />
                      </div>
                    </GlassCard>
                  </motion.div>
                );
              })}
            </div>
          )}
        </div>

        {/* Why Choose A1INC */}
        <div className="mb-40 grid grid-cols-1 lg:grid-cols-2 gap-20 items-center">
          <div>
            <h2 className="text-sm font-mono text-white/40 uppercase tracking-[0.2em] mb-4">The A1 Difference</h2>
            <h3 className="text-3xl md:text-4xl font-bold mb-8">What Makes Us Different</h3>
            <div className="space-y-6">
              {[
                { title: "Customized Solutions", desc: "Tailored plans based on your industry and goals." },
                { title: "Data-Driven Strategy", desc: "Every decision informed by insights and analytics." },
                { title: "Transparent Reporting", desc: "Clear communication and results tracking." },
                { title: "Full-Stack Support", desc: "Strategy → Execution → Optimization." }
              ].map((point, i) => (
                <div key={i} className="flex gap-4 p-6 rounded-2xl bg-white/[0.02] border border-white/5">
                  <div className="w-10 h-10 rounded-full bg-white/5 flex items-center justify-center shrink-0">
                    <CheckCircle2 className="w-5 h-5 text-white/40" />
                  </div>
                  <div>
                    <h4 className="font-bold mb-1">{point.title}</h4>
                    <p className="text-sm text-white/50">{point.desc}</p>
                  </div>
                </div>
              ))}
            </div>
          </div>
          <div className="relative aspect-square">
            <div className="absolute inset-0 bg-gradient-to-tr from-white/5 to-transparent rounded-3xl overflow-hidden">
               {/* Abstract visual placeholder */}
               <div className="absolute inset-0 flex items-center justify-center opacity-20">
                  <div className="w-64 h-64 border-2 border-white/10 rounded-full animate-[spin_20s_linear_infinite]" />
                  <div className="w-48 h-48 border border-white/20 rounded-full absolute animate-[spin_15s_linear_infinite_reverse]" />
               </div>
            </div>
          </div>
        </div>

        {/* Process Flow */}
        <div className="mb-40 pt-20 border-t border-white/5">
          <div className="text-center mb-16">
            <h2 className="text-3xl md:text-4xl font-bold mb-4">Our Proven Process</h2>
            <p className="text-white/50">A methodical approach to digital excellence.</p>
          </div>
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-8">
            {processSteps.map((step, i) => (
              <div key={i} className="relative p-8 rounded-3xl bg-white/[0.02] border border-white/5">
                <div className="absolute top-8 right-8 text-4xl font-display font-black text-white/[0.03]">0{i+1}</div>
                <h4 className="text-xl font-bold mb-2">{step.title}</h4>
                <p className="text-white/50 text-sm">{step.desc}</p>
              </div>
            ))}
          </div>
        </div>

        {/* Service Packages */}
        <div className="mb-40">
          <div className="text-center mb-16">
            <h2 className="text-3xl md:text-4xl font-bold mb-4">Pick a Package That Fits Your Goals</h2>
          </div>
          <div className="grid grid-cols-1 md:grid-cols-3 gap-8">
            {[
              { 
                name: "Starter", 
                price: "Ideal for launch", 
                includes: ["Website Setup", "Basic SEO", "Social Media Setup"],
                color: "bg-orange-500/10 border-orange-500/20" 
              },
              { 
                name: "Professional", 
                price: "Most Popular", 
                includes: ["Full Web Dev", "SEO", "Ads Management", "Content Strategy"],
                color: "bg-blue-500/10 border-blue-500/20",
                featured: true 
              },
              { 
                name: "Enterprise", 
                price: "Custom Suite", 
                includes: ["Everything + Custom Dev", "Branding", "Video", "Dedicated Manager"],
                color: "bg-purple-500/10 border-purple-500/20" 
              }
            ].map((pkg, i) => (
              <GlassCard key={i} className={`p-10 ${pkg.color} ${pkg.featured ? 'scale-105 ring-1 ring-white/20' : ''}`}>
                <h4 className="text-2xl font-bold mb-2">{pkg.name}</h4>
                <div className="text-sm font-mono text-white/40 mb-8">{pkg.price}</div>
                <ul className="space-y-4 mb-10">
                  {pkg.includes.map((item, j) => (
                    <li key={j} className="flex gap-3 text-sm text-white/60">
                      <CheckCircle2 className="w-4 h-4 shrink-0 text-white/20" />
                      {item}
                    </li>
                  ))}
                </ul>
                <Button className="w-full rounded-full" variant={pkg.featured ? "default" : "outline"}>
                  Get Started
                </Button>
              </GlassCard>
            ))}
          </div>
        </div>

        {/* Final CTA */}
        <GlassCard className="p-20 text-center bg-white/[0.02] border-white/5">
          <h2 className="text-4xl md:text-5xl font-bold mb-8">Ready to Elevate Your Digital Presence?</h2>
          <div className="flex flex-col sm:flex-row gap-4 justify-center">
            <Button size="lg" className="rounded-full px-8">Schedule a Consultation</Button>
            <Button size="lg" variant="outline" className="rounded-full px-8">See Our Portfolio</Button>
          </div>
        </GlassCard>
      </div>

      {/* Service Detail Modal */}
      <Dialog open={!!selectedService} onOpenChange={() => setSelectedService(null)}>
        <DialogContent className="max-w-2xl bg-black/95 border-white/10 text-white backdrop-blur-xl">
          <DialogHeader>
            <div className="w-12 h-12 rounded-xl bg-white/5 flex items-center justify-center mb-4">
              {selectedService && iconMap[selectedService.icon] && 
                (() => {
                  const Icon = iconMap[selectedService.icon];
                  return <Icon className="w-6 h-6" />;
                })()
              }
            </div>
            <DialogTitle className="text-3xl font-bold mb-2">{selectedService?.title}</DialogTitle>
            <div className="text-xs font-mono text-white/40 uppercase tracking-widest">{selectedService?.category}</div>
          </DialogHeader>
          
          <div className="mt-8 space-y-8">
            <div>
              <h4 className="text-sm font-mono text-white/40 uppercase tracking-widest mb-4">The Solution</h4>
              <p className="text-white/70 leading-relaxed text-lg">
                {selectedService?.longDescription || selectedService?.description}
              </p>
            </div>

            {selectedService?.features && (
              <div className="grid grid-cols-1 md:grid-cols-2 gap-8">
                <div>
                  <h4 className="text-sm font-mono text-white/40 uppercase tracking-widest mb-4">Core Features</h4>
                  <ul className="space-y-3">
                    {selectedService.features.map((f: string, i: number) => (
                      <li key={i} className="flex items-center gap-3 text-sm text-white/60">
                        <div className="w-1 h-1 rounded-full bg-white/40" />
                        {f}
                      </li>
                    ))}
                  </ul>
                </div>
                {selectedService?.benefits && (
                  <div>
                    <h4 className="text-sm font-mono text-white/40 uppercase tracking-widest mb-4">Key Benefits</h4>
                    <ul className="space-y-3">
                      {selectedService.benefits.map((b: string, i: number) => (
                        <li key={i} className="flex items-center gap-3 text-sm text-white/60">
                          <CheckCircle2 className="w-4 h-4 text-white/20" />
                          {b}
                        </li>
                      ))}
                    </ul>
                  </div>
                )}
              </div>
            )}

            <div className="pt-8 border-t border-white/5">
              <Button className="w-full h-14 rounded-2xl text-base font-bold">
                Inquire About This Service
              </Button>
            </div>
          </div>
        </DialogContent>
      </Dialog>
    </PageTransition>
  );
}
