import type { Express } from "express";
import type { Server } from "http";
import { storage } from "./storage";
import { api } from "@shared/routes";
import { z } from "zod";
import { insertContactSchema } from "@shared/schema";

export async function registerRoutes(
  httpServer: Server,
  app: Express
): Promise<Server> {
  
  // === Work Routes ===
  app.get(api.work.list.path, async (req, res) => {
    const work = await storage.getCaseStudies();
    res.json(work);
  });

  app.get(api.work.get.path, async (req, res) => {
    const slug = req.params.slug as string;
    const work = await storage.getCaseStudyBySlug(slug);
    if (!work) {
      return res.status(404).json({ message: "Case study not found" });
    }
    res.json(work);
  });

  // === Services Routes ===
  app.get(api.services.list.path, async (req, res) => {
    const services = await storage.getServices();
    res.json(services);
  });

  // === Contact Routes ===
  app.post(api.contact.submit.path, async (req, res) => {
    try {
      const input = insertContactSchema.parse(req.body);
      await storage.createContactSubmission(input);
      res.status(201).json({ success: true, message: "Message sent successfully" });
    } catch (err) {
      if (err instanceof z.ZodError) {
        return res.status(400).json({
          message: err.errors[0].message,
          field: err.errors[0].path.join('.'),
        });
      }
      throw err;
    }
  });

  // === Seed Data (Ideally this would be a separate script, but for Lite mode we'll do a quick check) ===
  await seedDatabase();

  return httpServer;
}

async function seedDatabase() {
  const existingWork = await storage.getCaseStudies();
  if (existingWork.length === 0) {
    console.log("Seeding database...");
    
    // Seed Work
    await storage.createCaseStudy({
      slug: "neon-fintech",
      title: "Neon Fintech",
      summary: "Redefining the banking experience with a futuristic glass UI.",
      role: "Design & Development",
      timeline: "3 Months",
      stack: ["React", "Three.js", "Node.js"],
      category: "Web",
      tags: ["Fintech", "Web3", "UI/UX"],
      thumbnail: "https://images.unsplash.com/photo-1551288049-bebda4e38f71?q=80&w=2670&auto=format&fit=crop",
      heroImage: "https://images.unsplash.com/photo-1551288049-bebda4e38f71?q=80&w=2670&auto=format&fit=crop",
      content: {
        problem: "The client needed a way to visualize complex financial data without overwhelming users.",
        approach: "We adopted a 'Glassmorphism' style to create depth and hierarchy.",
        solution: "A fully responsive, WebGL-powered dashboard.",
        results: "40% increase in user engagement."
      }
    });

    await storage.createCaseStudy({
      slug: "apex-architecture",
      title: "Apex Architecture",
      summary: "Minimalist portfolio for a high-end architectural firm.",
      role: "Full Stack",
      timeline: "2 Months",
      stack: ["Next.js", "Tailwind", "PostgreSQL"],
      category: "Branding",
      tags: ["Architecture", "Minimalism", "Portfolio"],
      thumbnail: "https://images.unsplash.com/photo-1487958449943-2429e8be8625?q=80&w=2670&auto=format&fit=crop",
      heroImage: "https://images.unsplash.com/photo-1487958449943-2429e8be8625?q=80&w=2670&auto=format&fit=crop",
      content: {
        problem: "Existing site was slow and cluttered.",
        approach: "Focus on imagery and typography.",
        solution: "A blazing fast static site with CMS integration.",
        results: "Awarded 'Site of the Day' on Awwwards."
      }
    });
    
    await storage.createCaseStudy({
        slug: "orbit-labs",
        title: "Orbit Labs",
        summary: "Brand identity system for a space-tech startup.",
        role: "Brand Identity",
        timeline: "4 Weeks",
        stack: ["Figma", "Illustrator", "Motion"],
        category: "Branding",
        tags: ["Space", "Identity", "Logo"],
        thumbnail: "https://images.unsplash.com/photo-1451187580459-43490279c0fa?q=80&w=2672&auto=format&fit=crop",
        heroImage: "https://images.unsplash.com/photo-1451187580459-43490279c0fa?q=80&w=2672&auto=format&fit=crop",
        content: {
          problem: "Brand lacked cohesion and modern appeal.",
          approach: "Create a system based on orbital mechanics.",
          solution: "A flexible identity system usable across all media.",
          results: "Successfully raised Series A funding."
        }
      });
      
    console.log("Seeding complete.");
  }
}
