import {
  type CaseStudy,
  type InsertCaseStudy,
  type ContactSubmission,
  type InsertContactSubmission,
  type Service
} from "@shared/schema";

export interface IStorage {
  // Work
  getCaseStudies(): Promise<CaseStudy[]>;
  getCaseStudyBySlug(slug: string): Promise<CaseStudy | undefined>;
  createCaseStudy(caseStudy: InsertCaseStudy): Promise<CaseStudy>;

  // Services
  getServices(): Promise<Service[]>;
  
  // Contact
  createContactSubmission(submission: InsertContactSubmission): Promise<ContactSubmission>;
}

export class MemStorage implements IStorage {
  private caseStudies: Map<number, CaseStudy>;
  private contactSubmissions: Map<number, ContactSubmission>;
  private services: Map<number, Service>;
  private currentIds: { [key: string]: number };

  constructor() {
    this.caseStudies = new Map();
    this.contactSubmissions = new Map();
    this.services = new Map();
    this.currentIds = { caseStudies: 1, contactSubmissions: 1, services: 1 };

    this.seed();
  }

  private seed() {
    const projects: InsertCaseStudy[] = [
      {
        slug: "tradetifs",
        title: "TradeTifs",
        category: "Trading Platform",
        summary: "Revolutionary B2B trading platform connecting global traders with premium commodities and investment opportunities.",
        role: "Full Stack Development",
        timeline: "6 Months",
        stack: ["React", "Node.js", "PostgreSQL", "TradingView"],
        tags: ["Trading", "Finance", "Enterprise"],
        thumbnail: "/src/assets/images/tradetifs.jpg",
        heroImage: "/src/assets/images/tradetifs.jpg",
        content: {
          sections: [
            { title: "Problem", body: "Traders lacked a unified platform for real-time commodity tracking and secure B2B transactions." },
            { title: "Approach", body: "We engineered a high-concurrency architecture with real-time websocket integration." },
            { title: "Outcome", body: "Successfully processed $20M+ in volume with 10K+ active traders globally." }
          ],
          highlights: ["10K+ Traders", "$20M+ Volume", "99.9% Uptime"]
        }
      },
      {
        slug: "tifs-academy",
        title: "TIFS Academy",
        category: "Learning Platform",
        summary: "Comprehensive online learning platform with professional courses taught by industry experts worldwide.",
        role: "UX Design & Development",
        timeline: "4 Months",
        stack: ["Next.js", "Tailwind", "Mux Video", "Stripe"],
        tags: ["Education", "Finance", "Trading"],
        thumbnail: "/src/assets/images/tifs-academy.jpg",
        heroImage: "/src/assets/images/tifs-academy.jpg",
        content: {
          sections: [
            { title: "Problem", body: "Legacy educational systems were clunky and failed to keep students engaged in complex financial topics." },
            { title: "Approach", body: "Gamified learning paths and seamless video streaming for an immersive experience." },
            { title: "Outcome", body: "Achieved a 4.9★ rating from over 7K+ active students worldwide." }
          ],
          highlights: ["7K+ Students", "3+ Courses", "4.9★ Rating"]
        }
      },
      {
        slug: "karbhaar",
        title: "Karbhaar",
        category: "Corporate Hub",
        summary: "Innovative corporate solutions platform connecting businesses with professional services and consulting tools.",
        role: "Branding & Strategy",
        timeline: "5 Months",
        stack: ["React", "Framer Motion", "Supabase"],
        tags: ["B2B", "Enterprise", "Services"],
        thumbnail: "/src/assets/images/karbhaar.jpg",
        heroImage: "/src/assets/images/karbhaar.jpg",
        content: {
          sections: [
            { title: "Problem", body: "Small and medium enterprises struggled to find reliable professional services in a fragmented market." },
            { title: "Approach", body: "Created a trust-based marketplace with verified pros and automated escrow." },
            { title: "Outcome", body: "Over $1M transacted within the first 6 months of launch." }
          ],
          highlights: ["1K+ Businesses", "5K+ Pros", "$1M Transacted"]
        }
      }
    ];

    projects.forEach(p => this.createCaseStudy(p));

    // Seed Services
    const serviceItems = [
      {
        slug: "web-development",
        title: "Web Development",
        description: "Custom, high-performance websites designed to convert and optimized for speed and SEO.",
        icon: "Code",
        category: "Engineering",
        longDescription: "Our web development process focuses on creating scalable, secure, and blazing-fast digital experiences. We use the latest technologies to ensure your site stays ahead of the curve.",
        features: ["Responsive Design", "Custom CMS Integration", "API Development", "Performance Optimization"],
        benefits: ["Faster load times", "Better search rankings", "Increased conversion rates"]
      },
      {
        slug: "social-media-management",
        title: "Social Media Management",
        description: "Strategic content planning and engagement to grow your digital footprint.",
        icon: "Monitor",
        category: "Marketing",
        longDescription: "Elevate your brand's presence across all social platforms. We handle everything from strategy and content creation to community management and reporting.",
        features: ["Content Strategy", "Platform Management", "Audience Engagement", "Analytics Reporting"],
        benefits: ["Brand consistency", "Community growth", "Higher engagement"]
      },
      {
        slug: "meta-google-ads",
        title: "Meta & Google Ads",
        description: "Precision ad campaigns that drive high-ROI and measurable results.",
        icon: "BarChart",
        category: "Marketing",
        longDescription: "Maximize your advertising budget with data-driven campaigns on Meta and Google. We target your ideal customers and optimize for the lowest possible cost per acquisition.",
        features: ["PPC Management", "Retargeting Campaigns", "Ad Creative Development", "A/B Testing"],
        benefits: ["Instant traffic", "Measurable ROI", "Targeted reach"]
      },
      {
        slug: "seo-optimization",
        title: "SEO Optimization",
        description: "Organic search strategies that help you rank higher and attract quality traffic.",
        icon: "Search",
        category: "Growth",
        longDescription: "Dominate the search results with our comprehensive SEO services. We cover technical SEO, on-page optimization, and strategic content creation.",
        features: ["Keyword Research", "Technical SEO Audits", "On-page Optimization", "Backlink Strategy"],
        benefits: ["Sustainable traffic", "Domain authority", "Reduced ad spend"]
      },
      {
        slug: "video-editing",
        title: "Video Editing",
        description: "Cinematic and compelling video content for all platforms.",
        icon: "Video",
        category: "Creative",
        longDescription: "Tell your story through high-quality video content. From social media reels to corporate documentaries, our editing team brings your vision to life.",
        features: ["Post-production", "Motion Graphics", "Color Grading", "Sound Design"],
        benefits: ["Higher retention", "Brand storytelling", "Social proof"]
      },
      {
        slug: "graphic-design",
        title: "Graphic Design",
        description: "Brand identity, visuals, and creative assets that resonate with audiences.",
        icon: "Paintbrush",
        category: "Creative",
        longDescription: "Stunning visual designs that capture your brand's essence. We create everything from logos and brand guidelines to marketing materials and social assets.",
        features: ["Logo Design", "Brand Identity", "Print Design", "Digital Assets"],
        benefits: ["Professional image", "Market recognition", "Visual impact"]
      }
    ];
    serviceItems.forEach((s, idx) => {
      this.services.set(idx + 1, { 
        ...s, 
        id: idx + 1,
        longDescription: s.longDescription ?? null,
        features: s.features ?? null,
        benefits: s.benefits ?? null
      });
    });
  }

  async getCaseStudies(): Promise<CaseStudy[]> {
    return Array.from(this.caseStudies.values());
  }

  async getCaseStudyBySlug(slug: string): Promise<CaseStudy | undefined> {
    return Array.from(this.caseStudies.values()).find(cs => cs.slug === slug);
  }

  async createCaseStudy(caseStudy: InsertCaseStudy): Promise<CaseStudy> {
    const id = this.currentIds.caseStudies++;
    const newCaseStudy: CaseStudy = { ...caseStudy, id, createdAt: new Date() };
    this.caseStudies.set(id, newCaseStudy);
    return newCaseStudy;
  }

  async getServices(): Promise<Service[]> {
    return Array.from(this.services.values());
  }

  async createContactSubmission(submission: InsertContactSubmission): Promise<ContactSubmission> {
    const id = this.currentIds.contactSubmissions++;
    const newSubmission: ContactSubmission = {
      ...submission,
      id,
      createdAt: new Date(),
      company: submission.company ?? null,
      budget: submission.budget ?? null
    };
    this.contactSubmissions.set(id, newSubmission);
    return newSubmission;
  }
}

export const storage = new MemStorage();
