import { pgTable, text, serial, timestamp, jsonb } from "drizzle-orm/pg-core";
import { createInsertSchema } from "drizzle-zod";
import { z } from "zod";

// === TABLE DEFINITIONS ===

// Case Studies for the "Work" section
export const caseStudies = pgTable("case_studies", {
  id: serial("id").primaryKey(),
  slug: text("slug").notNull().unique(),
  title: text("title").notNull(),
  summary: text("summary").notNull(),
  role: text("role").notNull(),
  timeline: text("timeline").notNull(),
  stack: text("stack").array().notNull(), // Array of strings for tech stack
  content: jsonb("content").notNull(), // Flexible content structure
  thumbnail: text("thumbnail").notNull(),
  heroImage: text("hero_image").notNull(),
  tags: text("tags").array().notNull(), // "Branding", "UI", etc.
  category: text("category").notNull(), // Main category for filtering
  createdAt: timestamp("created_at").defaultNow(),
});

// Contact Form Submissions
export const contactSubmissions = pgTable("contact_submissions", {
  id: serial("id").primaryKey(),
  name: text("name").notNull(),
  email: text("email").notNull(),
  company: text("company"),
  budget: text("budget"),
  message: text("message").notNull(),
  createdAt: timestamp("created_at").defaultNow(),
});

// Services (Optional, but good for dynamic rendering)
export const services = pgTable("services", {
  id: serial("id").primaryKey(),
  slug: text("slug").notNull().unique(),
  title: text("title").notNull(),
  description: text("description").notNull(),
  icon: text("icon").notNull(), // Icon name
  category: text("category").notNull(),
  longDescription: text("long_description"),
  features: text("features").array(),
  benefits: text("benefits").array(),
});

// === SCHEMAS ===

export const insertCaseStudySchema = createInsertSchema(caseStudies).omit({ id: true, createdAt: true });
export const insertContactSchema = createInsertSchema(contactSubmissions).omit({ id: true, createdAt: true });
export const insertServiceSchema = createInsertSchema(services).omit({ id: true });

// === TYPES ===

export type CaseStudy = typeof caseStudies.$inferSelect;
export type InsertCaseStudy = z.infer<typeof insertCaseStudySchema>;

export type ContactSubmission = typeof contactSubmissions.$inferSelect;
export type InsertContactSubmission = z.infer<typeof insertContactSchema>;

export type Service = typeof services.$inferSelect;

// Request/Response Types
export type CreateContactRequest = InsertContactSubmission;
